namespace Microsoft.Samples.PlanMyNight.Web.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.Linq;
    using System.Web.Mvc;
    using System.Web.Routing;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Infrastructure;
    using Microsoft.Samples.PlanMyNight.Infrastructure.Mvc;
    using Microsoft.Samples.PlanMyNight.Web.ViewModels;

    [HandleErrorWithContentType()]
    [OutputCache(NoStore = true, Duration = 0, VaryByParam = "*")]
    public class ItinerariesController : Controller
    {
        private readonly IItineraryContainer itineraryContainer;

        private readonly IItinerariesRepository itinerariesRepository;

        private readonly IActivitiesRepository activitiesRepository;

        public ItinerariesController() :
            this(
            new ServiceFactory().GetActivitiesRepositoryInstance(),
            new ServiceFactory().GetItinerariesRepositoryInstance(),
            new ServiceFactory().GetItineraryContainerInstance())
        {
        }

        public ItinerariesController(IActivitiesRepository activitiesRepository, IItinerariesRepository itinerariesRepository, IItineraryContainer itineraryContainer)
        {
            this.activitiesRepository = activitiesRepository;
            this.itinerariesRepository = itinerariesRepository;
            this.itineraryContainer = itineraryContainer;
        }

        [Authorize]
        public ActionResult Index()
        {
            var itineraries = this.itinerariesRepository.RetrieveByUser(this.User.UserId());
            var model = new ItinerariesViewModel
            {
                Itineraries = itineraries.Select(i => this.ToViewModel(i)).ToArray()
            };

            if (this.IsAjaxCall())
            {
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return View("Index", model);
        }

        public ActionResult ByUser(Guid id)
        {
            var itineries = this.itinerariesRepository.RetrieveByUser(id).Where(i => i.IsPublic);
            var model = new ItinerariesViewModel
            {
                Itineraries = itineries.Select(i => this.ToViewModel(i)).ToArray(),
                Owner = this.itinerariesRepository.GetUserDisplayName(id)
            };

            return View("ByUser", model);
        }

        public ActionResult Details(long id)
        {
            var itinerary = this.itinerariesRepository.Retrieve(id);

            // not public or not owner
            Guid userId = this.User.UserId();
            if (itinerary == null || (!itinerary.IsPublic && itinerary.UserId != userId))
            {
                return Redirect("~/");
            }

            // populate activity details
            this.activitiesRepository.PopulateItineraryActivities(itinerary);

            if (this.IsAjaxCall())
            {
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = this.ToViewModel(itinerary) };
            }

            // viewmodel
            var owner = this.itinerariesRepository.GetUserDisplayName(itinerary.UserId);
            var canEdit = (itinerary.UserId == userId);
            var canRate = userId != Guid.Empty && itinerary.IsPublic && this.itinerariesRepository.CanUserRateItinerary(id, userId);
            var model = this.ToViewModel(itinerary, owner, canEdit, canRate);

            return View("Details", model);
        }

        public ActionResult Current()
        {
            var itinerary = this.itineraryContainer.GetCurrent() ?? new Itinerary();
            this.activitiesRepository.PopulateItineraryActivities(itinerary);

            if (this.IsAjaxCall())
            {
                return new JsonResult
                {
                    JsonRequestBehavior = JsonRequestBehavior.AllowGet,
                    Data = this.ToViewModel(itinerary)
                };
            }

            return PartialView("Itinerary", this.ToViewModel(itinerary, string.Empty, true, true));
        }

        public ActionResult AddActivity(string activityId, int? index = null)
        {
            var itinerary = this.itineraryContainer.GetCurrent();
            if (itinerary == null)
            {
                itinerary = new Itinerary
                {
                    Name = string.Empty,
                    Created = DateTime.Now
                };
            }

            var existingActivity = itinerary.Activities.SingleOrDefault(a => a.ActivityId == activityId);
            if (existingActivity == null)
            {
                var activity = this.activitiesRepository.RetrieveActivity(activityId);
                if (activity != null)
                {
                    ReorderActivities(itinerary);
                    var nextOrder = 0;
                    if (itinerary.Activities.Count > 0) nextOrder = itinerary.Activities.Select(a => a.Order).Max() + 1;
                    itinerary.Activities.Add(new ItineraryActivity
                    {
                        Activity = activity,
                        ActivityId = activityId,
                        Order = nextOrder,
                        State = activity.State,
                        City = activity.City,
                        Zip = activity.Zip,
                        Longitude = activity.Location[0],
                        Latitude = activity.Location[1],
                        TypeId = activity.TypeId
                    });

                    this.itineraryContainer.SetCurrent(itinerary);

                    if (index.HasValue)
                    {
                        if (index.Value < itinerary.Activities.Count)
                        {
                            return this.MoveActivity(activityId, index.Value);
                        }
                    }
                }
            }

            this.activitiesRepository.PopulateItineraryActivities(itinerary);

            if (this.IsAjaxCall())
            {
                var model = this.ToViewModel(itinerary);
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return Redirect(this.GetReturnUrl());
        }

        public ActionResult AddActivities(string ids)
        {
            foreach (var id in ids.Split(new[] { '|' }, StringSplitOptions.RemoveEmptyEntries))
            {
                this.AddActivity(id);
            }

            if (this.IsAjaxCall())
            {
                var itinerary = this.itineraryContainer.GetCurrent();
                var model = this.ToViewModel(itinerary);
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return Redirect(this.GetReturnUrl());
        }

        public ActionResult DeleteActivity(string activityId)
        {
            var itinerary = this.itineraryContainer.GetCurrent();
            if (itinerary != null)
            {
                this.activitiesRepository.PopulateItineraryActivities(itinerary);
                var activity = itinerary.Activities.Where(a => a.ActivityId == activityId).SingleOrDefault();
                if (activity != null)
                {
                    itinerary.Activities.Remove(activity);
                    ReorderActivities(itinerary);
                    this.itineraryContainer.SetCurrent(itinerary);
                }
            }

            if (this.IsAjaxCall())
            {
                var model = this.ToViewModel(itinerary);
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return Redirect(this.GetReturnUrl());
        }

        public ActionResult MoveActivity(string activityId, int index)
        {
            var itinerary = this.itineraryContainer.GetCurrent();
            if (itinerary != null)
            {
                ReorderActivities(itinerary);
                var orderedActivities = itinerary.Activities.OrderBy(a => a.Order).ToList();
                var activity = orderedActivities.SingleOrDefault(a => a.ActivityId == activityId);
                if (activity != null)
                {
                    orderedActivities.Remove(activity);
                    orderedActivities.Insert(index, activity);
                    for (int i = 0; i < orderedActivities.Count; i++)
                    {
                        orderedActivities[i].Order = i;
                    }

                    this.itineraryContainer.SetCurrent(itinerary);
                }
            }

            if (this.IsAjaxCall())
            {
                var model = this.ToViewModel(itinerary);
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return Redirect(this.GetReturnUrl());
        }

        [ValidateAntiForgeryToken()]
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult New()
        {
            var itinerary = new Itinerary();
            this.itineraryContainer.SetCurrent(itinerary);

            if (this.IsAjaxCall())
            {
                var model = this.ToViewModel(itinerary, string.Empty, true, true);
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return Redirect(this.GetReturnUrl());
        }

        [Authorize]
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Edit(long id)
        {
            var itinerary = this.itinerariesRepository.Retrieve(id);
            if (itinerary == null || itinerary.UserId != this.User.UserId())
            {
                return Redirect("~/");
            }

            this.activitiesRepository.PopulateItineraryActivities(itinerary);
            this.itineraryContainer.SetCurrent(itinerary);

            // viewmodel
            var owner = this.itinerariesRepository.GetUserDisplayName(itinerary.UserId);
            var canEdit = (itinerary.UserId == this.User.UserId());
            var canRate = this.itinerariesRepository.CanUserRateItinerary(id, this.User.UserId());
            var model = this.ToViewModel(itinerary, owner, canEdit, canRate);

            return View("Edit", model);
        }

        [Authorize]
        [AcceptVerbs(HttpVerbs.Get)]
        public ActionResult Save()
        {
            var itinerary = this.itineraryContainer.GetCurrent();
            if (itinerary != null && itinerary.Activities.Count > 0)
            {
                var model = this.ToViewModel(itinerary, string.Empty, true, true);
                return View("Edit", model);
            }

            return RedirectToAction("Index");
        }

        [Authorize]
        [ValidateAntiForgeryToken()]
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult Save(string name, string description, bool? isPublic)
        {
            var itinerary = this.itineraryContainer.GetCurrent();
            if (itinerary != null && itinerary.Activities.Count > 0)
            {
                var userId = this.User.UserId();

                // TODO: Add validation: Name cannot be null
                itinerary.Name = name;
                if (description != null)
                {
                    itinerary.Description = description;
                }

                if (isPublic.HasValue)
                {
                    itinerary.IsPublic = isPublic.Value;
                }

                itinerary.Created = DateTime.Now;
                itinerary.UserId = userId;

                if (itinerary.Id != 0)
                {
                    this.itinerariesRepository.Update(itinerary);
                }
                else
                {
                    this.itinerariesRepository.Add(itinerary);
                }

                itinerary = this.itinerariesRepository.Retrieve(itinerary.Id);

                this.itineraryContainer.SetCurrent(itinerary);
                return RedirectToAction("Details", new { id = itinerary.Id });
            }

            return RedirectToAction("Index");
        }

        [Authorize]
        [ValidateAntiForgeryToken()]
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MakePublic(long id)
        {
            this.ChangePrivacyOptions(id, true);
            return RedirectToAction("Details", new { id = id });
        }

        [Authorize]
        [ValidateAntiForgeryToken()]
        [AcceptVerbs(HttpVerbs.Post)]
        public ActionResult MakePrivate(long id)
        {
            this.ChangePrivacyOptions(id, false);
            return RedirectToAction("Details", new { id = id });
        }

        [AcceptVerbs(HttpVerbs.Post)]
        [ValidateAntiForgeryToken()]
        public ActionResult SetEstimatedTime(string activityId, short estimatedMinutes)
        {
            var itinerary = this.itineraryContainer.GetCurrent();
            if (itinerary != null)
            {
                var activity = itinerary.Activities.SingleOrDefault(a => a.ActivityId == activityId);
                if (activity != null)
                {
                    activity.EstimatedMinutes = estimatedMinutes;
                    this.itineraryContainer.SetCurrent(itinerary);
                }
            }

            if (this.IsAjaxCall())
            {
                var model = this.ToViewModel(itinerary);
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            return Redirect(this.GetReturnUrl());
        }

        [Authorize]
        [AcceptVerbs(HttpVerbs.Post)]
        [ValidateAntiForgeryToken()]
        public ActionResult Rate(long id)
        {
            var rating = byte.Parse(Request.Form["Rating"], CultureInfo.InvariantCulture);
            if (rating > 0 && rating <= 5)
            {
                if (this.itinerariesRepository.CanUserRateItinerary(id, new Guid(User.Identity.Name)))
                {
                    this.itinerariesRepository.RateItinerary(id, new Guid(User.Identity.Name), rating, DateTime.Now);
                }
            }

            if (this.IsAjaxCall())
            {
                var updatedItinerary = this.itinerariesRepository.Retrieve(id);
                return new JsonResult
                {
                    JsonRequestBehavior = JsonRequestBehavior.AllowGet,
                    Data = new
                    {
                        UpdatedRatingLabel = ItineraryHelper.GetRatingLabel(updatedItinerary.Rating),
                        UpdatedRatingCss = ItineraryHelper.GetRatingCssClass(updatedItinerary.Rating) 
                    }
                };
            }
            else
            {
                return RedirectToAction("Details", new { id = id });
            }
        }

        [Authorize]
        [AcceptVerbs(HttpVerbs.Post)]
        [ValidateAntiForgeryToken()]
        public ActionResult AddComment(long id, string comment)
        {
            var itinerary = this.itinerariesRepository.Retrieve(id);
            var userId = this.User.UserId();
            if (itinerary != null && (itinerary.IsPublic || itinerary.UserId == userId))
            {
                var commentEntity = new ItineraryComment
                {
                    ItineraryId = id,
                    UserId = userId,
                    Body = comment,
                    Timestamp = DateTime.Now,
                    IpAddress = this.Request.ServerVariables["REMOTE_ADDR"]
                };

                this.itinerariesRepository.AddComment(commentEntity);

                return RedirectToAction("Details", "Itineraries", new { id = id, msg = "Thanks! Your comment has been published." });
            }

            return new RedirectResult(this.GetReturnUrl());
        }

        public PartialViewResult ListComments(long id)
        {
            var comments = this.itinerariesRepository.RetrieveComments(id);
            var model = new ItineraryCommentsViewModel
            {
                ItineraryId = id,
                Comments = comments
            };

            return PartialView("Comments", model);
        }

        private static void ReorderActivities(Itinerary itinerary)
        {
            int i = 0;
            foreach (var activity in itinerary.Activities.OrderBy(a => a.Order).ToArray())
            {
                activity.Order = i++;
            }
        }

        private void ChangePrivacyOptions(long id, bool toPublic)
        {
            var itinerary = this.itinerariesRepository.Retrieve(id);
            if (itinerary.UserId == this.User.UserId())
            {
                itinerary.IsPublic = toPublic;
                this.itinerariesRepository.Update(itinerary);

                // check if updating current itinerary
                var currentItinerary = this.itineraryContainer.GetCurrent();
                if (currentItinerary != null && currentItinerary.Id == itinerary.Id)
                {
                    currentItinerary.IsPublic = itinerary.IsPublic;
                    this.itineraryContainer.SetCurrent(currentItinerary);
                }
            }
        }
    }
}
